<?php
/**
 * Frontend Functions
 *
 * @package luxe
 * @since 1.0
*/
if(!function_exists('luxe_loader')) {
  /**
   * Loader
   *
   * @since luxe 1.0
  */
  function luxe_loader() { 
    $loader_enable = luxe_get_opt('general-loader-enable-switch');
    if(!$loader_enable && class_exists('ReduxFramework')) { return; } ?>
      <div class="preloader">
        <div class="preloader__bar"></div>
      </div>
    <?php
  }
}

if(!function_exists('luxe_main_menu')) {
  /**
   * Main Menu
   *
   * @since luxe 1.0
  */
  function luxe_main_menu($class = '', $menu_id = 'nav', $nav_menu = 'primary-menu') {
    $menu = luxe_get_opt('header-primary-menu');
    $menu = empty($menu) || !isset($menu) ? 'primary-menu':$menu;
    if ( function_exists('wp_nav_menu') && has_nav_menu( $nav_menu ) ) {
      wp_nav_menu(array(
        'theme_location' => $nav_menu,
        'container'      => false,
        'menu_id'        => $menu_id,
        'menu'           => $menu,
        'menu_class'     => $class,
        'walker'         => new luxe_menu_widget_walker_nav_menu()
      ));
    } else {
      echo '<ul id="nav" class="site-menu"><li class="menu-item"><a target="_blank" href="'. admin_url('nav-menus.php') .'" class="nav-list cell-view no-menu">'. esc_html__( 'You can edit your menu content on the Menus screen in the Appearance section.', 'luxe' ) .'</a></li></ul>';
    }
  }
}

if(!function_exists('luxe_top_header')) {
  /**
   * Top Header
   *
   * @since luxe 1.0
  */
  function luxe_top_header() { 
    $top_header_enable = luxe_get_opt('top-header-enable-switch'); 
    if(!$top_header_enable) { return; } 
    $top_header_phone    = luxe_get_opt('top-header-phone-number');
    $top_header_location = luxe_get_opt('top-header-location');
  ?>
    <div class="top-bar">
      <div class="container">
        <div class="top-bar__left">
          <div class="widget widget_contactinfo">
            <ul class="contact-info">
              <?php if(!empty($top_header_phone)): ?>
                <li class="contact-info-item">
                  <a href="#">
                    <i class="fa fa-phone"></i>
                    <?php echo esc_html($top_header_phone); ?>
                  </a>
                </li>
              <?php endif; ?>
              <?php if(!empty($top_header_location)): ?>
                <li class="contact-info-item">
                  <i class="fa fa-map-marker"></i> <?php echo esc_html($top_header_location); ?>
                </li>
              <?php endif; ?>
            </ul>
          </div>
        </div>
        <div class="top-bar__right">
          <div class="widget widget_social">
            <?php luxe_social_links('%s', luxe_get_opt('top-header-social-icons-category')); ?>
          </div>
        </div>
      </div>
    </div>
  <?php }
}

 if( !function_exists('luxe_logo')) {
  /**
   * Theme logo
   * @param string $logo_field
   * @param string $default_url
   * @param string $class
  */
  function luxe_logo($logo_field = '', $class = '',$default_url = '', $retina = false) {

    if (empty($logo_field)) {
      $logo_field = 'logo';
    }

    $logo = '';

    if( luxe_get_opt( $logo_field ) != null ) {
      $logo_array = luxe_get_opt( $logo_field );
    }

    if( (!isset( $logo_array['url'] ) || empty($logo_array['url'])) && empty($default_url)) {
      return;
    }

    if(empty($class)) {
      $class = ' logo vertical-align';
    }

    if( !isset( $logo_array['url'] ) || empty($logo_array['url']) ) {
      $logo_url = $default_url;
    } else {
      $logo_url = $logo_array['url'];
    }

    $width  = luxe_get_opt('logo-width');
    $height = luxe_get_opt('logo-height');


    $width_attr  = (!empty($width)) ? ' width="'.esc_attr($width).'"':'width="86"';
    $height_attr = (!empty($height)) ? ' height="'.esc_attr($height).'"':'height="27"';

    $style_attr = (!empty($height)) ? ' style="max-height:'.esc_attr($height).'px; height:auto;"':'style="max-height:27px;"';

    ?>
    <a href="<?php echo esc_url(home_url('/')); ?>" class="<?php echo luxe_sanitize_html_classes($class); ?>"><img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr(get_bloginfo( 'name' )); ?>"<?php echo wp_kses_post($width_attr); ?> <?php echo wp_kses_post($height_attr); ?><?php echo wp_kses_post($style_attr); ?>></a>
    <?php
  }
}


if ( ! function_exists( 'luxe_comment' ) ) :
/**
 * Comments and pingbacks. Used as a callback by wp_list_comments() for displaying the comments.
 *
 * @since make 1.0
 */
function luxe_comment( $comment, $args, $depth ) {
  $GLOBALS['comment'] = $comment;
  switch ( $comment->comment_type ) :
    case 'pingback' :
    case 'trackback' :
      ?>
      <li <?php comment_class('comment'); ?> id="li-comment-<?php comment_ID(); ?>">
        <div class="media-body"><?php esc_html_e( 'Pingback:', 'luxe' ); ?> <?php comment_author_link(); ?><?php edit_comment_link( __( 'Edit', 'luxe' ), ' ' ); ?></div>
      </li>
      <?php
    break;

    default :
      $class = array('comment_wrap');
      if ($depth > 1) {
        $class[] = 'chaild';
      }
      ?>
      <!-- Comment Item -->
      <li <?php comment_class('comment coment-block'); ?> id="comment-<?php comment_ID(); ?>">

        <div class="comment__wrap">
          <div class="comment__avatar">
            <?php echo get_avatar( $comment, 49 ); ?>
          </div>
          <div class="comment__header">
            <div class="comment__author"><?php comment_author_link();?></div>
            <time class="comment__date" datetime="<?php echo comment_date(get_option('date_format')) ?>"><?php echo comment_date(get_option('date_format')) ?></time>

            <?php $reply = get_comment_reply_link( array_merge( $args, array( 'depth' => $depth, 'max_depth' => 2 ) ) );
                if (!empty($reply)): ?>
                  <?php echo wp_kses_post($reply); ?>
                <?php endif;
                edit_comment_link( wp_kses_post( 'Edit', 'luxe' ), '', '' );
              ?>

          </div>
          <div class="comment__text">
            <?php if ( $comment->comment_approved == '0' ) : ?>
                <p><?php esc_html_e( 'Your comment is awaiting moderation.', 'luxe' ); ?></p>
            <?php endif; ?>
            <?php comment_text(); ?>
          </div>
        </div>
      <?php
      break;
  endswitch;
}

endif; // ends check for make_comment()

if (!function_exists('luxe_close_comment')):
/**
 * Close comment
 *
 * @since make 1.0
 */
function luxe_close_comment() { ?>
    </li>
    <!-- End Comment Item -->
<?php }

endif; // ends check for make_close_comment()



if( !function_exists('luxe_get_the_title')) {
  /**
   *
   * Get the Page Title
   * @since 1.0.0
   * @version 1.0.0
   *
  */
  function luxe_get_the_title() {
    $title = '';
    if (function_exists('is_woocoomerce') && is_woocommerce() || function_exists('is_shop') && is_shop()):
      if (apply_filters( 'woocommerce_show_page_title', true )):
        $title = woocommerce_page_title(false);
      endif;
    elseif( is_home() && !is_singular('page') ) :
      $title = esc_html__('Blog','luxe');

    elseif( is_singular() ) :
      $title = get_the_title();

    elseif( is_search() ) :
      global $wp_query;
      $total_results = $wp_query->found_posts;
      $prefix = '';

      if( $total_results == 1 ) {
        $prefix = esc_html__('1 search result for:', 'luxe');
      }
      else if( $total_results > 1 ) {
        $prefix = $total_results . ' ' . esc_html__('Search results for:', 'luxe');
      }
      else {
        $prefix = esc_html__('Search results for:', 'luxe');
      }
      $title = $prefix .' '. get_search_query();

    elseif ( is_category() ) :
      $title = single_cat_title('', false);

    elseif ( is_tag() ) :
      $title = single_tag_title('', false);

    elseif ( is_author() ) :
      $title = wp_kses_post(sprintf( __( 'Author: %s', 'luxe' ), '<span class="vcard">' . get_the_author() . '</span>' ));

    elseif ( is_day() ) :
      $title = wp_kses_post(sprintf( __( 'Day: %s', 'luxe' ), '<span>' . get_the_date() . '</span>' ));

    elseif ( is_month() ) :
      $title = wp_kses_post(sprintf( __( 'Month: %s', 'luxe' ), '<span>' . get_the_date( _x( 'F Y', 'monthly archives date format', 'luxe' ) ) . '</span>' ));

    elseif ( is_year() ) :
      $title = wp_kses_post(sprintf( __( 'Year: %s', 'luxe' ), '<span>' . get_the_date( _x( 'Y', 'yearly archives date format', 'luxe' ) ) . '</span>' ));

    elseif( is_tax() ) :
      $term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );
      $title = $term->name;

    elseif ( is_tax( 'post_format', 'post-format-aside' ) ) :
      $title = esc_html__( 'Asides', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-gallery' ) ) :
      $title = esc_html__( 'Galleries', 'luxe');

    elseif ( is_tax( 'post_format', 'post-format-image' ) ) :
      $title = esc_html__( 'Images', 'luxe');

    elseif ( is_tax( 'post_format', 'post-format-video' ) ) :
      $title = esc_html__( 'Videos', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-quote' ) ) :
      $title = esc_html__( 'Quotes', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-link' ) ) :
      $title = esc_html__( 'Links', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-status' ) ) :
      $title = esc_html__( 'Statuses', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-audio' ) ) :
      $title = esc_html__( 'Audios', 'luxe' );

    elseif ( is_tax( 'post_format', 'post-format-chat' ) ) :
      $title = esc_html__( 'Chats', 'luxe' );

    elseif( is_404() ) :
      $title = esc_html__( '404', 'luxe' );

    else :
      $title = esc_html__( 'Archives', 'luxe' );
    endif;

    return $title;
  }
}

if(!function_exists('luxe_breadcrumbs')) {
  /**
   * Show breadcrumbs
   * 
   * @since luxe 1.0
   */
  function luxe_breadcrumbs() { 

    $breadcrumbs = luxe_get_opt('title-wrapper-breadcrumb-enable');

    if(class_exists('ReduxFramework') && !$breadcrumbs) { return; }
    
    $before      = '<li class="breadcrumbs__item">';
    $after       = '</li>';
    $before_last = '';
    $after_last  = '';
    $separator = '';
    ?>
    <!-- Breadcrumbs -->
    <ul class="breadcrumbs">
      <?php
      
      if (function_exists('is_woocoomerce') && is_woocommerce() || function_exists('is_shop') && is_shop()) {
        $args = array(
          'delimiter'   => '&nbsp;/&nbsp;',
          'wrap_before' => '',
          'wrap_after'  => '',
          'before'      => '',
          'after'       => '',
          'home'        => esc_html__( 'Home', 'luxe' )
        );

        woocommerce_breadcrumb($args);

      } else if (!is_home()) {

        echo wp_kses_post($before).'<a href="';
        echo esc_url(home_url('/'));
        echo '" class="breadcrumbs__link">';
        echo __('Home', 'luxe');
        echo '</a>'.$after. $separator .'';
        
        if (is_single()) {

          $cats = get_the_category();

          if( isset($cats[0]) ) :
            echo wp_kses_post($before.'<a href="'. esc_url(get_category_link( $cats[0]->term_id )) .'">'. $cats[0]->cat_name.'</a>' . $after . $separator);
          endif;

          if (is_single()) {
            echo wp_kses_post($before.$before_last);
            the_title();
            echo wp_kses_post($after_last.$after);
          }
        } elseif( is_category() ) {

          $cats = get_the_category();

          if( isset($cats[0]) ) :
            echo wp_kses_post($before.$before_last.single_cat_title('', false).$after_last.$after);
          endif;

        } elseif (is_page()) {
          global $post;
          
          if($post->post_parent){
            $anc = get_post_ancestors( $post->ID );
            $title = get_the_title();
            foreach ( $anc as $ancestor ) {
              $output = $before.'<a href="'.esc_url(get_permalink($ancestor)).'" title="'.esc_attr(get_the_title($ancestor)).'"  rel="v:url" property="v:title">'.get_the_title($ancestor).'</a>'.$after.' ' . $separator;
            }
            echo wp_kses_post($output);
            echo wp_kses_post($before.$before_last.$title.$after_last.$after);
          } else {
            echo wp_kses_post($before.$before_last.get_the_title().$after_last.$after);
          }
        }
        elseif (is_tag()) { 
          echo wp_kses_post($before.$before_last).single_cat_title('', false).$after_last.$after; 
          
        } elseif (is_day()) {
          echo wp_kses_post($before.$before_last). esc_html__('Archive for', 'luxe').' '; 
          the_time('F jS, Y');
          echo wp_kses_post($after_last.$after);
          
        } elseif (is_month()) {
          echo wp_kses_post($before.$before_last).esc_html__('Archive for', 'luxe').' '; 
          the_time('F, Y');
          echo wp_kses_post($after_last.$after);
          
        } elseif (is_year()) {
          echo wp_kses_post($before.$before_last). esc_html__('Archive for', 'luxe').' '; 
          the_time('Y');
          echo wp_kses_post($after_last.$after);
          
        } elseif (is_author()) {
          echo wp_kses_post($before.$before_last). esc_html__('Author Archive', 'luxe');
          echo wp_kses_post($after_last.$after);
          
        } elseif (isset($_GET['paged']) && !empty($_GET['paged'])) { 
          echo wp_kses_post($before.$before_last).esc_html__('Blog Archives', 'luxe').$after_last.$after;
          
        } elseif (is_search()) {
          echo wp_kses_post($before.$before_last). esc_html__('Search Results', 'luxe').$after_last.$after;
          
        }

      } elseif (is_home()) { 
        echo wp_kses_post($before.$before_last). esc_html__('Home', 'luxe') .$after_last.$after; 
      }
      ?>
    </ul>
    <!-- End Breadcrumbs -->
  <?php }
}

if(!function_exists('luxe_blog_post_format')) {
  /**
   *
   * Post Format
   * @since 1.0.0
   * @version 1.0.0
   *
  */
  function luxe_blog_post_format() {
    $post_format = get_post_format();
    switch ($post_format) {
      case 'audio':
        $audio_id  = luxe_get_post_opt('post-audio-id');
        if(!empty($audio_id)):
          echo (is_single()) ? '<div class="block"><div class="block__content">':''; ?>
            <div class="sound-item">
              <iframe src="https://w.soundcloud.com/player/?url=https%3A//api.soundcloud.com/tracks/<?php echo esc_attr($audio_id); ?>&amp;color=ff5500&amp;auto_play=false&amp;hide_related=false&amp;show_comments=true&amp;show_user=true&amp;show_reposts=false"></iframe>
            </div>
          <?php echo (is_single()) ? '</div></div>':''; ?>
        <?php endif;
        break;
      case 'video':
        global $post;
        $img_src        = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'medium-big');
        $video_url_m4v  = luxe_get_post_opt('post-video-url-m4v');
        $video_url_webm = luxe_get_post_opt('post-video-url-webm');
        if(isset($img_src[0]) && !empty($video_url_m4v) && !empty($video_url_webm)): ?>
          <div class="video-item">
            <div class="bg" style="background-image:url(<?php echo esc_url($img_src[0]); ?>)"></div>
            <div class="play-button"><i class="icon-play"></i></div>
             <div class="video-wrapper">
             <video loop class="bgvid">
              <source type="video/webm" src="<?php echo esc_url($video_url_webm); ?>" />
              <source type="video/mp4" src="<?php echo esc_url($video_url_m4v); ?>" />
             </video>
             <div class="close-video"><span>+</span></div>
             </div>
          </div>
        <?php endif;
        break;
      case 'gallery': 
        $post_gallery = luxe_get_post_opt('post-gallery');
        $image_size   = (is_single()) ? 'luxe-large':'luxe-medium';
        echo (is_single()) ? '<div class="block"><div class="block__content">':''; 
      ?>
        <div class="image-gallery">
          <div class="image-gallery__main">
            <?php 
              if(is_array($post_gallery) && !empty($post_gallery)):
                foreach ( $post_gallery as $key => $gallery ): 
                  $image_url = luxe_get_image_src($gallery['attachment_id'], $image_size);
            ?>
                <div class="image-gallery__item"><img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($gallery['title']); ?>"></div>
            <?php endforeach; endif; ?>
          </div>
          <?php if(is_single() && is_array($post_gallery) && !empty($post_gallery)): ?>
            <ul class="image-gallery__thumbs">
              <?php foreach ( $post_gallery as $key => $gallery ): $thumb_url = luxe_get_image_src($gallery['attachment_id'], 'luxe-thumb'); ?>
                  <li class="image-gallery__thumb"><button class="image-gallery__button"><img src="<?php echo esc_url($thumb_url); ?>" alt="<?php echo esc_attr($gallery['title']); ?>"></button></li>
              <?php endforeach; ?>
          </ul>
          <?php endif; ?>
        </div>
        <?php echo (is_single()) ? '</div></div>':''; ?>
      <?php      
        break;
      
      default:
        if(has_post_thumbnail() && !is_single()): ?>
          <div class="image text-center">
            <a href="<?php echo esc_url(get_the_permalink()); ?>"><?php the_post_thumbnail('luxe-medium'); ?></a>
          </div>
        <?php endif;
        break;
    }
  }
}

if(!function_exists('luxe_social_links')) {
  /**
   * Get Social Icons links
   *
   * @param type $terms
   * @return boolean
  */
  function luxe_social_links($pattern = '%s', $category = '') {
    $args = array(
      'posts_per_page' => -1,
      'offset'          => 0,
      'orderby'         => 'menu_order',
      'order'           => 'ASC',
      'post_type'       => 'social-site',
      'post_status'     => 'publish'
    );

    if (!empty($category)) {
      $args['tax_query'] = array(
        array(
          'taxonomy' => 'social-site-category',
          'field'    => 'id',
          'terms'    => $category,
        ),
      );
    }

    $custom_query = new WP_Query( $args );
    if ( $custom_query->have_posts() ):

      $found_posts = $custom_query->found_posts;
      echo '<ul class="social">';
      while ( $custom_query -> have_posts() ) :
        $custom_query -> the_post();
        echo sprintf($pattern, '<li class="social-item"><a class="social-link" target="_blank" href="'.esc_url(get_the_title()).'"><i class="fa '.luxe_get_post_opt('icon').'"></i></a></li>');
      endwhile; // end of the loop.
      echo '</ul>';
    endif;
    wp_reset_postdata();
  }
}

if(!function_exists('luxe_footer_columns')) {
  /**
   * Footer Columns
   * @param type $type
   * @return array
  */
  function luxe_footer_columns() { 
    $footer_columns = luxe_get_opt('footer-column');
    $footer_columns = (class_exists('ReduxFramework')) ? $footer_columns:4;
    switch ($footer_columns) {
      case '1':
        $col_class = 'col-md-12 col-sm-12';
        break;
      case '2':
        $col_class = 'col-md-6 col-sm-6';
        break;
      case '3':
        $col_class = 'col-md-4 col-sm-6';
        break;
      case '4':
      default:
        $col_class = 'col-md-3 col-sm-6';
        break;
    }
    for($i = 1; $i < $footer_columns + 1; $i++) { ?>
      <div class="<?php echo esc_attr($col_class .' col-'.$i); ?>">
        <?php if (is_active_sidebar( luxe_get_custom_sidebar('footer-'.$i, 'footer-sidebar-'.$i) )): ?>
          <?php dynamic_sidebar( luxe_get_custom_sidebar('footer-'.$i, 'footer-sidebar-'.$i) ); ?>
        <?php endif; ?>
        <div class="empty-space marg-xs-b30"></div>
      </div>
    <?php }
  }
}

if(!function_exists('luxe_wrapper_start')) {
  /**
   * luxe_wrapper_start
   * @param type $type
   * @return array
  */
  function luxe_wrapper_start() { 
    echo '<div class="outer-wrapper_layout">';
  }
}

if(!function_exists('luxe_wrapper_end')) {
  /**
   * luxe_wrapper_end
   * @param type $type
   * @return array
  */
  function luxe_wrapper_end() { 
    echo '</div>';
  }
}

if(!function_exists('luxe_accent_css')) {
  function luxe_accent_css() {
    $accent_color = luxe_get_opt('theme-skin-accent');
    $output = '';
    if(luxe_get_opt('theme-skin') == 'theme-accent' && !empty($accent_color) && class_exists('ReduxFramework')):
      $output .= '
      .blog-nav a:hover,
      .blog-nav span:hover,
      .sticky .recent-post,
      .tab--style-2 .tab-nav__item.active .tab-nav__link,
      .cs-check,
      .checkbox-select input[type="checkbox"]:checked,
      .checkbox-select input[type="checkbox"].error:checked,
      #hotel-booking-payment input[type="text"]:focus,
      #hotel-booking-payment .hb-addition-information textarea:focus,
      #commentform #text:focus,
      #commentform #name:focus,
      #commentform #email:focus,
      #commentform #website:focus,
      #hotel-booking-payment input[type="email"]:focus,
      #review_form_wrapper textarea:focus,
      .hb_single_room #reviews #review_form_wrapper form input:focus {
        border-color:'.esc_attr($accent_color).';
      }

      .blog-nav a:hover,
      .blog-nav span:hover,
      .site-menu .sub-menu a:hover,
      .site-menu .sub-menu a:focus,
      .site-footer .tnp-submit,
      .btn:not(.btn--style-2),
      .hb_single_room #reviews #review_form_wrapper form .form-submit input[type="submit"],
      .icon--style-2:hover,
      .icon--style-2:focus,
      .tab--style-2 .tab-nav__item.active .tab-nav__link,
      .sidebar .contact-info__icon,
      .irs-bar,
      .search-filter__submit,
      .ui-datepicker.ui-widget .ui-datepicker-calendar .ui-state-default:hover,
      .hb_mini_cart_footer a,
      #hotel-booking-cart .hb_button.hb_checkout,
      #hotel-booking-payment button[type="submit"],
      #hotel-booking-payment button[type="button"],
      #hotel-booking-cart .hb_button.hb_checkout, .hero-tabs .btn,
      .hb-video__close:hover:before,.hb-video__close:hover:after,
      .hb_mini_cart_footer a:hover, #hotel-booking-cart .hb_button.hb_checkout:hover, #hotel-booking-payment button[type="submit"]:hover,
      #hotel-booking-payment button[type="button"]:hover,
      .header-btn, .hb-no__result, .widget_hb_widget_search .hb-submit button[type="submit"],
      .hb-extra_btn,.hb-extra_btn:hover, .wp-hotel-booking-search-rooms button[type="submit"],
      #hotel-booking-cart button#hb-apply-coupon, #hotel-booking-payment button#hb-apply-coupon,
      #hotel-booking-cart button[type="submit"].update, #hotel-booking-cart button[type="submit"].update:hover {
        background:'.esc_attr($accent_color).';
      }

      .rating-input.high-light:before,
      .input--checkbox input:checked+label::after,
      .btn--site-2,
      .btn--site-2:hover,
      .btn--site-2:focus,
      .featured-rooms__link,
      .icon__media,
      .list__item::before,
      .list-icon__item::before,
      .star-rating,
      .room-sm__price em,
      .widget-layered-nav-list__item--chosen,
      .recent-post__meta a,
      .rating-input:before,
      .hotel_booking_mini_cart .hb_mini_cart_item .hb_mini_cart_price span,
      .hotel_booking_mini_cart .hb_mini_cart_item .hb_mini_cart_remove:hover {
        color:'.esc_attr($accent_color).';
      }
      ';
    endif;

    $width  = luxe_get_opt('logo-width');
    $height = luxe_get_opt('logo-height');
    if(!empty($width) || !empty($height)):
      $output .= '.site-logo a img {';
      $output .= (!empty($width)) ? 'width:'.esc_attr($width).'px;':'';
      $output .= (!empty($height)) ? 'height:'.esc_attr($height).'px;':'';
      $output .= '}';
    endif;

    $logo_container_height = luxe_get_opt('logo-container-height');
    if(!empty($logo_container_height)):
      $output .= '@media (min-width: 768px) {';
      $output .= '.above-menu>.container {';
      $output .= 'height:'.esc_attr($logo_container_height).'px;';
      $output .= '}';
      $output .= '}';
    endif;


    $logo_container_bg = luxe_get_opt('customizer-logo-container-bg');
    if(!empty($logo_container_bg)):
      $output .= '@media (min-width: 992px) {';
      $output .= '.hb-header__style4 .nav-bar__item.nav-bar__item--logo .site-logo {';
      $output .= (!empty($logo_container_bg)) ? 'background:'.esc_attr($logo_container_bg).';':'';
      $output .= '}';
      $output .= '}';
    endif;


    return $output;
  }
}